<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\SimBridge;
use Illuminate\Http\Request;
use Carbon\Carbon;

class SimBridgeController extends Controller
{
    public function register(Request $request)
    {
        $request->validate([
            'device_name' => 'required|string',
            'sim_number' => 'required|string|unique:sim_bridges',
            'device_info' => 'required|json'
        ]);

        $simBridge = SimBridge::create([
            'device_name' => $request->device_name,
            'sim_number' => $request->sim_number,
            'ip_address' => $request->ip(),
            'device_info' => $request->device_info,
            'sim_expires_at' => Carbon::now()->addDays(30), // Default 30 days
            'last_ping' => Carbon::now()
        ]);

        return response()->json(['success' => true, 'data' => $simBridge]);
    }

    public function updateStatus(Request $request)
    {
        $request->validate([
            'sim_number' => 'required|exists:sim_bridges,sim_number',
            'data_used' => 'required|integer',
            'status' => 'required|in:active,inactive,expired'
        ]);

        $simBridge = SimBridge::where('sim_number', $request->sim_number)->first();
        $simBridge->update([
            'data_used' => $request->data_used,
            'status' => $request->status,
            'last_ping' => Carbon::now()
        ]);

        return response()->json(['success' => true]);
    }

    public function ping(Request $request)
    {
        $request->validate([
            'sim_number' => 'required|exists:sim_bridges,sim_number'
        ]);

        $simBridge = SimBridge::where('sim_number', $request->sim_number)->first();
        $simBridge->update(['last_ping' => Carbon::now()]);

        return response()->json(['success' => true]);
    }

    public function getStats()
    {
        $stats = SimBridge::active()
            ->select('id', 'device_name', 'data_used', 'weight')
            ->get()
            ->map(function ($bridge) {
                $bridge->load_percentage = $bridge->weight;
                return $bridge;
            });

        return response()->json(['success' => true, 'data' => $stats]);
    }

    public function index()
    {
        $simBridges = SimBridge::orderBy('created_at', 'desc')->get();
        return response()->json(['success' => true, 'data' => $simBridges]);
    }

    public function updateWeight(Request $request)
    {
        $request->validate([
            'weights' => 'required|array',
            'weights.*.id' => 'required|exists:sim_bridges,id',
            'weights.*.weight' => 'required|integer|min:0|max:100'
        ]);

        foreach ($request->weights as $weight) {
            SimBridge::where('id', $weight['id'])
                ->update(['weight' => $weight['weight']]);
        }

        return response()->json(['success' => true]);
    }

    public function destroy(SimBridge $simBridge)
    {
        $simBridge->delete();
        return response()->json(['success' => true]);
    }
}