<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ConnectionLog;
use App\Models\VpnPayload;
use Illuminate\Http\Request;
use Carbon\Carbon;

class ConnectionLogController extends Controller
{
    public function connect(Request $request)
    {
        $request->validate([
            'server_name' => 'required|string',
            'protocol' => 'required|in:tcp,udp'
        ]);

        if (!auth()->user()->canConnect()) {
            return response()->json([
                'success' => false,
                'message' => 'Connection not allowed. Please check your subscription or trial status.'
            ], 403);
        }

        $connection = ConnectionLog::create([
            'user_id' => auth()->id(),
            'server_name' => $request->server_name,
            'protocol' => $request->protocol,
            'ip_address' => $request->ip(),
            'connected_at' => Carbon::now()
        ]);

        return response()->json([
            'success' => true,
            'data' => $connection
        ]);
    }

    public function disconnect(Request $request)
    {
        $activeConnection = auth()->user()->activeConnections()->first();
        
        if (!$activeConnection) {
            return response()->json([
                'success' => false,
                'message' => 'No active connection found'
            ], 404);
        }

        $activeConnection->update([
            'disconnected_at' => Carbon::now(),
            'bytes_sent' => $request->bytes_sent ?? 0,
            'bytes_received' => $request->bytes_received ?? 0
        ]);

        return response()->json(['success' => true]);
    }

    public function status()
    {
        $activeConnection = auth()->user()->activeConnections()->first();
        
        return response()->json([
            'success' => true,
            'connected' => (bool) $activeConnection,
            'data' => $activeConnection
        ]);
    }

    public function history(Request $request)
    {
        $logs = auth()->user()->connectionLogs()
            ->orderBy('connected_at', 'desc')
            ->paginate($request->per_page ?? 15);

        return response()->json([
            'success' => true,
            'data' => $logs
        ]);
    }
}